/* FrequencyList.java (iterative version)
 * CS 136 Spring 2007
 * Tutorial 10 - List Processing, Generics, and Iteration
 */

/* A frequency list is a list of items along with their frequencies.
 * For this implementation, the generic type parameter T represents the
 * type of item whose frequencies we are counting, and we define the
 * frequency of item T to be the number of times add(T) has been called.
 * (More precisely, the number of times add(T2) has been called with
 * T2.equals(T).)
 * 
 * The underlying data structure for the implementation is a GenList (like
 * a Scheme list) of FrequencyPair<T> objects, each of which stores an item
 * of type T and an integer frequency.
 */
public class FrequencyList<T> {
  /* The list of items and frequencies, stored in any order. */
  private GenList<FrequencyPair<T>> flist;
  
  /* Creates a new frequency list, initially empty. */
  public FrequencyList() {
    flist = null;
  }
  
  /* Checks if this frequency list is empty.
   * pre: true
   * post: true
   * return: true iff this frequency list is empty.
   */
  public boolean empty() {
    return null == flist;
  }
  
  /* Gets the length of this frequency list.
   * pre: true
   * post: true
   * return: The number of distinct items that have been
   *         added to this frequency list.
   */
  public int length() {
    int len = 0;
    GenList<FrequencyPair<T>> lst = flist;
    
    while( lst != null ) {
      ++len;
      lst = lst.rest;
    }
    
    return len;
  }
  
  /* Adds an item to this frequency list.
   * pre: item != null
   * post: If item has not been added before, it is
   *       added to the frequency list with frequency 1.
   *       Otherwise, the frequency of item in the list
   *       is increased by 1.
   */
  public void add (T item) {
    GenList<FrequencyPair<T>> lst = flist;

    while( lst != null ) {
      if( lst.first.getItem().equals(item) ) {
        lst.first.incrFreq();
        return;
      }
      lst = lst.rest;
    }
    
    flist = new GenList<FrequencyPair<T>>( new FrequencyPair<T>(item), flist );
  }
  
  /* Gets the most frequent item in the list.
   * pre: !this.empty()
   * post: true
   * return: The most frequently added item.
   *         If there is more than one item with the highest
   *         frequency, any one of them is returned.
   */
  public T mostFrequent() {
    FrequencyPair<T> soFar = flist.first;
    GenList<FrequencyPair<T>> lst = flist.rest;
    
    while( lst != null ) {
      if( soFar.lessFrequentThan( lst.first ) )
        soFar = lst.first;
      lst = lst.rest;
    }
    
    return soFar.getItem();
  }
  
  /* Gets the k most frequently added items, along with their frequencies.
   * pre: !this.empty() && (k > 0)
   * post: true
   * return: A GenList of FrequencyPair<T>s containing the k items with
   *         the highest frequencies, sorted from least to most frequent.
   *         If this.length() < k, then the length of the returned list
   *         is this.length(); otherwise it is k.
   */
  public GenList<FrequencyPair<T>> mostFrequent( int k ) {
    GenList<FrequencyPair<T>> soFar = new GenList<FrequencyPair<T>>(flist.first,null);
    GenList<FrequencyPair<T>> lst = flist.rest;
    
    for( int i = 1; (i < k) && (lst != null); ++i ) {
      soFar = insert( lst.first, soFar );
      lst = lst.rest;
    }
    
    while( lst != null ) {
      if( soFar.first.lessFrequentThan( lst.first ) )
        soFar = insert( lst.first, soFar.rest );
      lst = lst.rest;
    }
    
    return soFar;
  }
  
  /* Helper method for mostFrequent(int k) which inserts a given item into
   * the given list so that the result is sorted by increasing frequency.
   * pre: (item != null && lst != null) and item is not in lst.
   * post: true
   * return: A GenList of FrequencyPair<T>s with all the items in lst,
   *         plus item, sorted from least to most frequent.
   */
  private GenList<FrequencyPair<T>> insert( FrequencyPair<T> item, GenList<FrequencyPair<T>> lst ) {
    if( item.lessFrequentThan( lst.first ) )
      return new GenList<FrequencyPair<T>>(item,lst);
    
    GenList<FrequencyPair<T>> cur = lst;
    while( cur.rest != null ) {
      if( item.lessFrequentThan( cur.rest.first ) ) {
        cur.rest = new GenList<FrequencyPair<T>>(item,cur.rest);
        return lst;
      }
      cur = cur.rest;
    }
    
    cur.rest = new GenList<FrequencyPair<T>>(item,null);
    return lst;
  }

}
