/* GradeBook.java
 * CS 136 Spring 2007
 * Tutorial 12: Double Arrays and Polymorphism
 */

/* This class stores a list of students and their marks on
 * some number of assignments (at most 100).
 */
public class GradeBook {
  // The students' names
  private String[] students;
  
  // The double-array of marks. The first index is the student,
  // and the second index is the assignment number.
  private int[][] marks;
  
  // The maximum scores for the assignments
  private int[] max;
  
  // The number of assignments entered so far
  private int nAssignments;
  
  /* Creates a new gradebook from the given list of student names.
   * pre: studentNames != null, each name in studentNames is not null
   */
  public GradeBook( String[] studentNames ) {
    students = new String[studentNames.length];
    for( int i = 0; i < students.length; ++i )
      students[i] = studentNames[i];
    
    marks = new int[students.length][100];
    max = new int[100];
    
    nAssignments = 0;
  }
  
  /* Gets the number of assignments entered so far.
   * pre: true
   * post: true
   * return: the number of assignments entered so far.
   */
  public int numAssignments() { return nAssignments; }
  
  /* Gets the number of students in the gradebook
   * pre: true
   * post: true
   * return: the number of students in the gradebook
   */
  public int numStudents() { return students.length; }
  
  /* Adds a new assignment to the gradebook.
   * pre: scores != null, each score in scores is at least 0,
   *      scores.length = (number of students in gradebook)
   *      maxScore > 0, numAssignments() < 100
   * post: A new assignment is added with the given scores for
   *       the corresponding students.
   */
  public void addAssignment( int[] scores, int maxScore ) {
    for( int i = 0; i < students.length; ++i )
      marks[i][nAssignments] = scores[i];
    max[nAssignments] = maxScore;
    ++nAssignments;
  }
  
  /* Gets a single mark in the gradebook.
   * pre: student is the name of a student in the gradebook,
   *      0 <= asst < numAssignments()
   * post: true
   * return: The mark that the named student got on the
   *         indicated assignment.
   */
  public int getMark( String student, int asst ) {
    int sind = findStudent(student);
    assert( sind >= 0 );
    return marks[sind][asst];
  }
  
  /* Changes a single mark in the gradebook.
   * pre: student is the name of a student in the gradebook,
   *      0 <= asst < numAssignments(), and newMark >= 0
   * post: The mark for the given student and the given assignment index
   *       is set to the value of newMark.
   */
  public void changeMark( String student, int asst, int newMark ) {
    int sind = findStudent(student);
    assert( sind >= 0 );
    marks[sind][asst] = newMark;
  }
  
  /* Gets the average mark for a given assignment.
   * pre: 0 <= asst < numAssignments()
   * post: true
   * return: The average mark for all the students on the specified assignment.
   */
  public double avgMark( int asst ) {
    int sum = 0;
    for( int i = 0; i < students.length; ++i )
      sum += marks[i][asst];
    return ((double)sum)/students.length;
  }
  
  /* Gets the average percentage for the named student over
   * all assignments.
   * pre: name is the name of a student in the handbook
   * post: true
   * return: The named student's score on each assignment is
   *         scaled from 0-1 (according to the corresponding
   *         max score), and the average of these scaled values
   *         is returned.
   */
  public double studentAvg( String name ) {
    int sind = findStudent(name);
    assert(sind >= 0 );
    return studentAvg(sind);
  }
  
  /* Gets the student with the highest average according to studentAvg.
   * pre: true
   * post: true
   * return: The name of the student with the highest value
   *         from the method studentAvg.
   */
  public String topStudent() {
    int topInd = 0;
    double topAvg = studentAvg(0);
    for( int i = 1; i < students.length; ++i ) {
      double avg = studentAvg(i);
      if( avg > topAvg ) {
        topAvg = avg;
        topInd = i;
      }
    }
    return students[topInd];
  }
  
  /* Curves the specified assignment to the highest grade.
   * pre: 0 <= asst < numAssignments()
   * post: All students' scores on the specified assignment are
   *       multiplied by the same constant (possibly rounded down)
   *       so that at least one student has the maximum score (and
   *       none have more than the maximum).
   */
  public void curveAssignment( int asst ) {
    int m = 0;
    for( int i = 0; i < students.length; ++i ) {
      if( marks[i][asst] > m ) m = marks[i][asst];
    }
    
    for( int i = 0; i < students.length; ++i ) {
      marks[i][asst] *= max[asst];
      marks[i][asst] /= m;
    }
  }
  
  /* Helper method for studentAvg(String) which takes
   * the index of a student and returns that student's
   * average (scaled) score.
   * pre: 0 <= index < (number of students)
   * post:
   * return: The indicated student's average.
   */
  private double studentAvg( int index ) {
    double sum = 0;
    for( int i = 0; i < nAssignments; ++i ) {
      sum += ((double)marks[index][i])/max[i];
    }
    return sum/nAssignments;
  }
  
  /* Searches for the given student name.
   * pre: true
   * post: true
   * return: The index of the given name in the students array,
   *         or -1 if the name is not found.
   */
  private int findStudent( String name ) {
    for( int i = 0; i < students.length; ++i ) {
      if( students[i].equals(name) ) return i;
    }
    return -1;
  }
}