/* newinterp.h
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Black-box interpolation over a finite field using diversification.
 * See "Diversification improves interpolation", Giesbrecht & Roche, 2011
 *
 * Header file
 */

#ifndef NEWINTERP_H
#define NEWINTERP_H

#include <NTL/ZZ.h>
#include "blackbox.h"

/* Computes the least d>0 so that the given polynomial f
 * is (e,d)-diverse. That is, all coefficients of f with
 * absolute value at least d have pairwise differences at
 * least 2e.
 */
template <typename T>
T diversity_bound (const std::vector< std::complex<T> >& f, T e);

/* Computes an approximation for the 2-norm of f,
 * provided f has degree at most D and at most T terms.
 * BB should be a subclass of ApproxBB
 */
template <typename BB>
typename BB::FloatT approxnorm (BB& bb, const NTL::ZZ& D, long T);

/* Probabilistic interpolation method "A".
 * This is a mostly straightforward implementation of the method in
 * "Diversification improves interpolation", Giesbrecht & Roche, 2011,
 * made somewhat more adaptive by combining all "for" loops into one.
 * The verification step is not included; the algorithm is Monte Carlo.
 * BBT: should be a subclass of ApproxBB
 * f: will hold the output
 * bb: approximate black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: false if a known failure occurred, otherwise true
 */
template <typename BBT>
bool new_interpA (SparsePoly< typename BBT::FloatT >& f, 
                  BBT& bb, const NTL::ZZ& D, long T);

#include "newinterp.inc"

#endif // NEWINTERP_H
