/* GSinterp.h
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Black-box interpolation over a finite field using the algorithm
 * of Garg & Schost (2009)
 *
 * Header file
 */

#ifndef GSINTERP_H
#define GSINTERP_H

#include <NTL/ZZ.h>
#include "blackbox.h"

NTL_OPEN_NNS

/* Computes an upper bound on the coefficients in Chi. */
void chi_bound (ZZ &out, const ZZ& D, long T);

/* Deterministic interpolation method.
 * This is the original algorithm from the paper.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: f as a SparsePoly
 */
template <typename BBT>
void gs_determ_interp 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T);

/* Probabilistic interpolation method "A".
 * This slight modification randomly samples primes that are "large enough",
 * rather than examining all small primes up to a bound.
 * The result is a Monte Carlo deterministic algorithm that should be faster.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: f as a SparsePoly
 */
template <typename BBT>
void gs_prob_interpA 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T);

/* Probabilistic interpolation method "B".
 * This further modification only samples one "large enough" prime,
 * then searches for a bunch of small good ones.
 * Provable worst-case complexity is worse than method A, but
 * should be better in practice by a factor of log deg f or so.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: f as a SparsePoly
 */
template <typename BBT>
void gs_prob_interpB 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T);

NTL_CLOSE_NNS

#include "GSinterp.inc"

#endif // GSINTERP_H
