/* GSinterp.inc
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Black-box interpolation over a finite field using the algorithm
 * of Garg & Schost (2009)
 *
 * Include file (template implementations)
 */

#include <cmath>
#include <vector>
#include <utility>
#include <algorithm>
#include <NTL/ZZX.h>
#include <NTL/lzz_pX.h>
#include <NTL/ZZXFactoring.h>
#include "misc.h"

NTL_OPEN_NNS

// For STL sorting of pointers to factors
inline static bool factor_sort (const ZZX* a, const ZZX* b) 
    { return ConstTerm(*a) > ConstTerm(*b); } 

/* Interpolates an unknown sparse polynomial from a set of
 * sufficiently many "good prime" evaluations modulo x^p - 1.
 * First evaluate chi = \prod_{1 <= i <= t} (x-e_i),
 * then factor it to extract exponents, and finally match those
 * with coefficients.
 * f: will hold the output
 * t: The true sparsity of the unknown f
 * evals: A list of pairs (p, fp) such that each fp has sparsity t
 *        and the product of p's is large enough to recover chi
 */
template <typename Poly, typename Base>
static void gs_evals_interp 
  (SparsePoly<Poly,Base>& f, long t, 
   const std::vector< std::pair<long, Poly> >& evals)
{
  zz_pBak bak; // for switching the NTL global modulus
  ZZX chi; // Integer polynomial with roots at f's exponents
  zz_pX chip; // will hold each chi mod p
  ZZ pprod = to_ZZ(1);

  // Create two vectors to hold the exponents mod p
  std::vector<long> expons(t);
  vec_zz_p ntl_expons;
  ntl_expons.SetLength(t);

  for (typename std::vector< std::pair<long, Poly> >::const_iterator
       iter = evals.begin();
       iter != evals.end();
       ++iter)
  {
    // Extract the exponents of nonzero terms
    long i = 0;
    for (long j=0; j<=deg(iter->second); ++j) {
      if (!IsZero(coeff(iter->second,j)))
        expons[i++] = j;
    }
    assert (i == t);

    // Now we have to change the global NTL prime by "pushing" onto the "stack"
    bak.save();
    zz_p::init(iter->first);
    // Now copy expons to ntl_expons
    for (long i=0; i<t; ++i)
      conv (ntl_expons[i], expons[i]);
    // Initialize chip from the roots
    BuildFromRoots (chip, ntl_expons);
    // Update chi with the new modular image at each coefficient
    CRT (chi, pprod, chip);
    bak.restore(); // "pop" our prime off the "stack"
  }

  // Now we have chi to full precision, so we need its roots, which
  // are the exponents of nonzero terms in f
  vec_ZZX factors;
  SFFactor (factors, chi);
  assert (factors.length() == t);
  
  // We want to get the exponents in sorted order.
  // So make a vector of pointers to each factor, then sort it.
  // (This is to avoid directly sorting a vector of ZZ's, which
  //  would involve more potentially costly copying.)
  std::vector<ZZX*> facptrs(t);
  for (long i=0; i<t; ++i) facptrs[i] = &(factors[i]);
  std::sort (facptrs.begin(), facptrs.end(), factor_sort );

  // Now extract the exponents from factors and store them in f.
  // Simultaneously, use a single good-prime evaluation to
  // get the coefficients.
  f.rep.resize(t);
  typename SparsePoly<Poly,Base>::RepT::iterator fiter = f.rep.begin();
  for (std::vector<ZZX*>::const_iterator eiter = facptrs.begin();
       eiter != facptrs.end(); ++eiter) {
    negate (fiter->second, ConstTerm(**eiter));
    fiter->first = 
      coeff (evals[0].second, rem(fiter->second,evals[0].first));
    ++fiter;
  }
}

/* Computes an upper bound on the coefficients in Chi. */
void chi_bound (ZZ &out, const ZZ& D, long T) {
  out = D;
  out += 1;
  for (ZZ i = D; i >= D - (T-2); --i) out *= i;
}

/* Deterministic interpolation method.
 * This is the original algorithm from the paper.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: f as a SparsePoly
 */
template <typename BBT>
void gs_determ_interp 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T)
{
  // Compute qbound, upper bound on coefficients in Chi
  ZZ qbound;
  chi_bound (qbound, D, T);

  // Compute lower bound s on the number of bad primes
  long s = (long) floor (log(D) * T * (T-1) * .5 / std::log(2.0));

  // Use NTL's precomputed list of the first primes and repeatedly make
  // black box calls. t is the current best estimate for the sparsity,
  // and goodp, goodfp hold black box evaluations with sparsity t.
  // Once the product of primes in goodp is larger than qbound AND
  // at least s primes have been examined, we move on.
  std::vector< std::pair<long, typename BBT::PolyT> > goodevals;
  ZZ goodprod; // product of good primes
  conv(goodprod,1);
  typename BBT::PolyT fp;
  long t = 0;
  long p = 1; 
  long sfp;
#ifdef VERBOSE
  long ngood = 0;
  long nbad = 0;
#endif

  // Polynomials for the black box evaluation
  typename BBT::PolyT x;
  SetCoeff(x,1);
  typename BBT::PolyT xpm1;
  typename BBT::BaseT minusone;
  conv (minusone, -1);
  SetCoeff (xpm1, 0, minusone);
  typename BBT::BaseT zero;
  conv (zero, 0);

  PrimeSeq ps;
  while (s > 0 || (goodprod <= qbound)) {
    p = ps.next();
    assert (p != 0);
    SetCoeff (xpm1, p);
    bb.eval(fp, x, xpm1);
    SetCoeff (xpm1, p, zero);
    sfp = sparsity(fp);
    if (sfp > t) {
#ifdef VERBOSE
      nbad += goodevals.size();
#endif
      goodevals.clear();
      conv(goodprod,1);
      t = sfp;
    }
    if (sfp == t && (goodprod <= qbound)) {
      goodevals.resize (goodevals.size()+1);
      goodevals.back().first = p;
      goodevals.back().second = fp;
      goodprod *= p;
    }
#ifdef VERBOSE
    else if (sfp == t) ++ngood;
    else ++nbad;
#endif
    --s;
  }
#ifdef VERBOSE
  std::cout << "Sampled mod x^p-1 for " 
            << ngood+goodevals.size() << " good primes and "
            << nbad << " bad primes" << std::endl
            << "in the range 2 <= p <= "
            << p << std::endl;
  std::cout << (ngood ? "Only " : "All ") << goodevals.size()
            << " good prime samples were actually used." << std::endl;
#endif

  gs_evals_interp (f, t, goodevals);
}

/* Probabilistic interpolation method A.
 * This slight modification randomly samples primes that are "large enough",
 * rather than examining all small primes up to a bound.
 * The result is a Monte Carlo deterministic algorithm that should be faster.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: f as a SparsePoly
 */
template <typename BBT>
void gs_prob_interpA 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T)
{
  // Compute qbound, upper bound on coefficients in Chi
  ZZ qbound;
  chi_bound (qbound, D, T);

  // Compute lambda s.t. a prime in the range lambda..2*lambda is good w.h.p.
  long lambda = goodp_bound (D, T);
  long goodplen = NumBits(lambda) + 1;

  // Use NTL to generate random primes and repeatedly make
  // black box calls. t is the current best estimate for the sparsity,
  // and goodevals holds black box evaluations with sparsity t.
  // Once the product of primes in goodp is larger than qbound,
  // we move on.
  std::vector< std::pair<long, typename BBT::PolyT> > goodevals;
  ZZ goodprod; // product of good primes
  conv(goodprod,1);
  typename BBT::PolyT fp;
  long t = 0;
  long p; 
  long sfp;
#ifdef VERBOSE
  long nbad = 0;
#endif

  // Polynomials for the black box evaluation
  typename BBT::PolyT x;
  SetCoeff(x,1);
  typename BBT::PolyT xpm1;
  typename BBT::BaseT minusone;
  conv (minusone, -1);
  SetCoeff (xpm1, 0, minusone);
  typename BBT::BaseT zero;
  conv (zero, 0);

  while (goodprod <= qbound) {
    do { p = GenPrime_long (goodplen); } while (divide(goodprod,p));
    SetCoeff (xpm1, p);
    bb.eval(fp, x, xpm1);
    SetCoeff (xpm1, p, zero);
    sfp = sparsity(fp);
    if (sfp > t) {
#ifdef VERBOSE
      nbad += goodevals.size();
#endif
      goodevals.clear();
      conv(goodprod,1);
      t = sfp;
    }
    if (sfp == t) {
      goodevals.resize (goodevals.size()+1);
      goodevals.back().first = p;
      goodevals.back().second = fp;
      goodprod *= p;
    }
#ifdef VERBOSE
    else ++nbad;
#endif
  }
#ifdef VERBOSE
  std::cout << "Sampled mod x^p-1 for " 
            << goodevals.size() << " good primes and "
            << nbad << " bad primes" << std::endl
            << "in the range " << (1L<<(goodplen-1)) << " <= p <= "
            << ((1L<<goodplen)-1) << std::endl;
#endif

  gs_evals_interp (f, t, goodevals);
}

/* Probabilistic interpolation method "B".
 * This further modification only samples one "large enough" prime,
 * then searches for a bunch of small good ones.
 * Provable worst-case complexity is worse than method A, but
 * should be better in practice by a factor of log deg f or so.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: f as a SparsePoly
 */
template <typename BBT>
void gs_prob_interpB 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T)
{
  // Compute qbound, upper bound on coefficients in Chi
  ZZ qbound;
  chi_bound (qbound, D, T);

  // Compute lambda s.t. a prime in the range lambda..2*lambda is good w.h.p.
  long lambda = goodp_bound (D, T);
  long goodplen = NumBits(lambda) + 1;

  // Compute a guess as to where we should start searching for good primes
  long guess = guessp_bound (D, T);

  // Use NTL to generate random primes and repeatedly make
  // black box calls. t is the current best estimate for the sparsity,
  // and goodevals holds black box evaluations with sparsity t.
  // Once the product of primes in goodp is larger than qbound,
  // we move on.
  std::vector< std::pair<long, typename BBT::PolyT> > goodevals(1);
  ZZ goodprod; // product of good primes
  typename BBT::PolyT fp;
  long t;
  long p; 
  long sfp;
#ifdef VERBOSE
  long nbad = 0;
#endif

  // Polynomials for the black box evaluation
  typename BBT::PolyT x;
  SetCoeff(x,1);
  typename BBT::PolyT xpm1;
  typename BBT::BaseT minusone;
  conv (minusone, -1);
  SetCoeff (xpm1, 0, minusone);
  typename BBT::BaseT zero;
  conv (zero, 0);

  // First we do one "big prime" evaluation
  // which gets the sparsity t correct w.h.p.
  p = GenPrime_long (goodplen);
  conv(goodprod,p);
  SetCoeff (xpm1, p);
  bb.eval(goodevals.front().second, x, xpm1);
  SetCoeff (xpm1, p, zero);
  t = sparsity(goodevals.front().second);
  goodevals.front().first = p;
#ifdef VERBOSE
  long firstp = p;
#endif

  PrimeSeq ps;
  ps.reset (guess);
  while (goodprod <= qbound) {
    p = ps.next();
    SetCoeff (xpm1, p);
    bb.eval(fp, x, xpm1);
    SetCoeff (xpm1, p, zero);
    sfp = sparsity(fp);
    if (sfp > t) {
#ifdef VERBOSE
      nbad += goodevals.size();
#endif
      goodevals.clear();
      conv(goodprod,1);
      t = sfp;
    }
    if (sfp == t) {
      goodevals.resize (goodevals.size()+1);
      goodevals.back().first = p;
      goodevals.back().second = fp;
      goodprod *= p;
    }
#ifdef VERBOSE
    else ++nbad;
#endif
  }
#ifdef VERBOSE
  std::cout << "Sampled mod x^p-1 for a single ";
  if (firstp == goodevals.front().first) {
    std::cout << "good prime p=" << firstp << ',' << std::endl
              << "as well as " << goodevals.size()-1 << " good and ";
  }
  else {
    std::cout << "BAD prime p=" << firstp << ',' << std::endl
              << "as well as " << goodevals.size() << " good and ";
  }
  std::cout << nbad << " bad primes" << std::endl
            << "in the range " << guess << " <= p <= "
            << p << std::endl;
#endif

  gs_evals_interp (f, t, goodevals);
}

NTL_CLOSE_NNS


