/* blackbox.h
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Classes for a uni-modular black box
 * See "Diversification improves interpolation", Giesbrecht & Roche, 2011
 *
 * These classes are designed to be used with NTL types, for instance
 * zz_p for Base and zz_pX for Poly.
 *
 * Header file
 */

#ifndef BLACKBOX_H
#define BLACKBOX_H

#include "sparsepoly.h"

NTL_OPEN_NNS

/* Base class for a uni-modular black box.
   Represents evaluating an unknown polynomial modulo a chosen g.
 */
template <typename Poly, typename Base>
class UniModBB {
  public:
    typedef Poly PolyT;
    typedef Base BaseT;
    /* Returns the unknown polynomial f evaluated at f(h) modulo g.
     * Must have deg h < deg g.
     */
    virtual void eval (Poly& out, const Poly& h, const Poly& g) = 0;
};


/* Fake uni-modular black box.
 * Constructed with a reference to a SparsePoly
 */
template <typename Poly, typename Base>
class FakeUniModBB : public UniModBB<Poly,Base> {
  private:
    const SparsePoly<Poly,Base> f;
  public:
    FakeUniModBB (const SparsePoly<Poly,Base>& thef)
      :f(thef) { }

    /* Returns the "unknown" polynomial evaluated at h modulo g */
    void eval (Poly& out, const Poly& h, const Poly& g);
};

NTL_CLOSE_NNS

#include "blackbox.inc"

#endif // BLACKBOX_H
