/* blackbox.inc
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Classes for a uni-modular black box
 * See "Diversification improves interpolation", Giesbrecht & Roche, 2011
 *
 * These classes are designed to be used with NTL types, for instance
 * zz_p for Base and zz_pX for Poly.
 *
 * Include file (template implementations)
 */

#include <cassert>
#include <vector>
#include <NTL/ZZ.h>

NTL_OPEN_NNS

/* Returns the "unknown" polynomial evaluated at h modulo g */
template <typename Poly, typename Base>
void FakeUniModBB<Poly,Base>::eval (Poly& out, const Poly& h, const Poly& g) {
  clear(out);
  std::vector<Poly> hpows(1); // hpows[i] = h^(2^i) mod g
  hpows[0] = h;
  Poly curhpow;
  SetCoeff(curhpow,0,1); // curhpow = 1
  ZZ curpow; // invariant: curhpow = h^curpow mod g
  ZZ diffpow;

  typename SparsePoly<Poly,Base>::RepT::const_iterator
    iter = f.rep.begin();

  if (IsZero(iter->second)) {
    SetCoeff(out, 0, iter->first);
    ++iter;
  }

  while (iter != f.rep.end()) {
    diffpow = iter->second;
    diffpow -= curpow;
    while (static_cast<unsigned long>(NumBits(diffpow)) > hpows.size()) {
      // grow hpows if necessary for this step
      hpows.resize(hpows.size()+1);
      SqrMod (hpows[hpows.size()-1], hpows[hpows.size()-2], g);
    }
    while (!IsZero(diffpow)) {
      MulMod (curhpow, curhpow, hpows[NumBits(diffpow)-1], g);
      SwitchBit (diffpow, NumBits(diffpow)-1);
    }
    out += curhpow * iter->first;
    curpow = iter->second;
    ++iter;
  }
}

NTL_CLOSE_NNS


