/* misc.cc
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Miscellaneous utilities for the interpolation code.
 * Implementation file
 */

#include <cmath>
#include <iostream>
#include <fstream>
#include <NTL/ZZ.h>
#include "misc.h"

NTL_START_IMPL

/* Seed NTL's random generator from an unsigned long long */
void ullseed (unsigned long long seed) {
  SetSeed (ZZFromBytes (reinterpret_cast<unsigned char*>(&seed),
                        sizeof(unsigned long long)));
}

/* Initialize NTL's random seed from /dev/urandom and
 * (optionally) print its value to stdout.
 */
void randseed (bool printit) {
  unsigned long long seed;
  ifstream randin ("/dev/urandom");
  //FILE *randin = fopen ("/dev/urandom", "r");
  randin.read ((char*)(&seed), sizeof(unsigned long long)); 
  //fread (&seed, sizeof(unsigned long long), 1, randin);
  randin.close();
  //fclose (randin);
  ullseed (seed);
  if (printit) {
    std::cout << "--------------------------------------------------------------------------------" << std::endl 
              << "The pseudo-random number generator has been seeded from /dev/urandom." << std::endl
              << "If something interesting happens and you want to re-create it, replace" << std::endl
              << "the call to randseed() with "
              << "ullseed(0x" << std::hex << seed << std::dec
              << "ULL);" << std::endl
              << "--------------------------------------------------------------------------------" << std::endl;
  }
}

/* Computes a lower bound on lambda so that at least half the primes
 * between lambda and 2*lambda are good.
 * See Lemma 3.1 from Giesbrecht & Roche (2011).
 */
long goodp_bound (const ZZ& D, long T) {
  long lambda = (long) std::ceil 
    ((5.0/3.0) * T * (T-1) * log(D));
  if (lambda < 21) lambda = 21;
  return lambda;
}

/* Computes a "best guess" bound on a small number to start searching
 *  for good primes.
 * This bound is based on the birthday problem, assuming the exponents
 * are randomly distributed in the range [0,...,p-1] (they are not!), 
 * how large does p have to be in order to make them all distinct w.h.p.
 * The returned value is based on the exponential approximation to the
 * birthday problem.
 */
long guessp_bound (const ZZ& D, long T) {
  double dubT = (double) T;
  return std::ceil(dubT*(dubT-1)/std::log(4.0));
}

NTL_END_IMPL
