/* test-ck.c
 * Daniel S. Roche, January 2009
 * http://www.cs.uwaterloo.ca/~droche/
 *
 * Test code for classical/Karatsuba polynomial multiplication.
 * Two filenames should be given as arguments, for the input and output.
 * The input has six integers on the first line:
 * t: the number of test cases to follow
 * m: greater than the degree of the first polynomial in each pair
 * n: greater than the degree of the second polynomial in each pair
 * p: the modulus to use
 * k, w: Ignored integers
 * What follows are 2t lines each containing a polynomial 
 * (alternating sizes m and n),
 * formatted as [a0 a1 a2 ... am] where a0..am are the coefficients.
 * The output consists of t polynomials (the products), in the same format,
 * one per line.
 * Upon completion, the number of seconds used to 
 * perform the computation is written to standard out.
 *
 * See LICENSE.txt for copyright and permissions.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include "lsmul.h"

int main(int argc, char** argv) {
   long t,m,n,i,mi,ni;
   long *sizes;
   lsmul_ele p,w;
   int k;
   lsmul_ele *polys, *currF;
   clock_t cstart, cend;
   FILE *in, *out;

   assert(argc == 3);
   in = fopen(argv[1],"r");
   assert(in);

   i = fscanf(in, " %ld %ld %ld " LSMUL_READ_ELE " %d " LSMUL_READ_ELE, 
      &t, &m, &n, &p, &k, &w);
   assert (i == 6);

   polys = malloc ((2*m + 2*n - 1) * t * sizeof(lsmul_ele));
   assert (polys);
   sizes = malloc (2*t*sizeof(long));
   assert (sizes);

   // Read in the test data
   currF = polys;
   for (i=0; i<2*t; i += 2) {
      sizes[i] = read_poly_prealloc (in, currF);
      sizes[i+1] = read_poly_prealloc (in, currF + sizes[i]);
      currF += 2*sizes[i] + 2*sizes[i+1] - 1;
   }

   fclose(in);

   // Perform the multiplications
   cstart = clock();
   currF = polys;
   for(i=0; i<2*t; i += 2) {
      mi = sizes[i];
      ni = sizes[i+1];

      mul_ck (currF+mi+ni, currF, mi, currF+mi, ni, p);

      currF += 2*mi + 2*ni - 1;
   }
   cend = clock();

   // Write the results
   out = fopen(argv[2],"w");
   assert(out);
   currF = polys;
   for (i=0; i<2*t; i += 2) {
      mi = sizes[i];
      ni = sizes[i+1];
      write_poly (out, currF+mi+ni, mi+ni-1);
      putc( '\n', out );
      currF += 2*mi + 2*ni - 1;
   }
   fclose(out);

   printf("========================================================================\n");
   printf("Time for c/k: %lf\n", ((double)(cend-cstart))/CLOCKS_PER_SEC);
   printf("========================================================================\n");

   free(sizes);
   free(polys);

   return 0;
}
