/* test-fft.c
 * Daniel S. Roche, January 2009
 * http://www.cs.uwaterloo.ca/~droche/
 *
 * Test code for FFT-based polynomial multiplication.
 * Two filenames should be given as arguments, for the input and output.
 * The input has six integers on the first line:
 * t: the number of test cases to follow
 * m: greater than the degree of the first polynomial in each pair
 * n: greater than the degree of the second polynomial in each pair
 * p: the modulus to use
 * k: an integer such that 2^k >= m+n-1
 * w: a 2^k-PRU in Z/pZ
 * What follows are 2t lines each containing a polynomial 
 * (alternating sizes m and n),
 * formatted as [a0 a1 a2 ... am] where a0..am are the coefficients.
 * The output consists of t polynomials (the products), in the same format,
 * one per line.
 * Upon completion, the number of seconds used to perform the computation
 * is written to standard out.
 *
 * See LICENSE.txt for copyright and permissions.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include "lsmul.h"

int main(int argc, char** argv) {
   long t,m,n,sc,i,mi,ni;
   long *sizes;
   lsmul_ele p,w;
   int k;
   lsmul_ele *polys, *currF;
   clock_t cstart, cend;
   FILE *in, *out;

   assert(argc == 3);
   in = fopen(argv[1],"r");
   assert(in);

   i = fscanf(in, " %ld %ld %ld " LSMUL_READ_ELE " %d " LSMUL_READ_ELE, 
      &t, &m, &n, &p, &k, &w);
   assert (i == 6);

   i = m + n - 2;
   sc = 2;
   while (i >>= 1) sc <<= 1;

   //polys = malloc ((m + n + sc) * t * sizeof(lsmul_ele));
   polys = malloc ((3*sc) * t * sizeof(lsmul_ele));
   assert (polys);
   sizes = malloc (2*t*sizeof(long));
   assert (sizes);

   // Read in the test data
   currF = polys;
   for (i=0; i<2*t; i += 2) {
      sizes[i] = read_poly_prealloc (in, currF);
      sizes[i+1] = read_poly_prealloc (in, currF + sizes[i]);
      currF += sizes[i] + sizes[i+1] + sc;
   }

   fclose(in);

   // Perform the multiplications
   cstart = clock();
   currF = polys;
   for(i=0; i<2*t; i += 2) {
      mi = sizes[i];
      ni = sizes[i+1];

      mul_fft (currF+mi+ni, currF, mi, currF+mi, ni, p, k, w);

      currF += mi + ni + sc;
   }
   cend = clock();

   // Write the results
   out = fopen(argv[2],"w");
   assert(out);
   currF = polys;
   for (i=0; i<2*t; i += 2) {
      mi = sizes[i];
      ni = sizes[i+1];
      write_poly (out, currF+mi+ni, mi+ni-1);
      putc( '\n', out );
      currF += mi + ni + sc;
   }
   fclose(out);

   printf("========================================================================\n");
   printf("Time for FFT: %lf\n", ((double)(cend-cstart))/CLOCKS_PER_SEC);
   printf("========================================================================\n");

   free(sizes);
   free(polys);

   return 0;
}
