/* pp.h
 * Dan Roche, 21 Jan 2007
 * http://www.cs.uwaterloo.ca/~droche/
 * 
 * Given an integer polynomial f, our goal is to find an r such 
 * that f=h^r and r>1, or determine that none exists.
 */

#ifndef PP_H
#define PP_H

#include <NTL/ZZX.h>
#include <utility>

/* Sanity check/control
 * Uses square-free decomposition routines from NTL.
 * This method is deterministic (always correct) but (somewhat) slow.
 */
long pp_sqd( const NTL::ZZX& f );

/* New method to detect lacunary polynomials that are perfect
 * powers, as presented in Giesbrecht & Roche, ISSAC 2008 (submitted).
 * The polynomial input must be in the lacunary representation
 * (see denseToSparse below), along with its degree and the number
 * of bits in the infinity norm
 * (we could of course compute these from the representation directly,
 *  but that would waste precious time). 
 * The integer pbound is used to choose primes.
 */
long pp_lacunary( const std::pair<NTL::vec_ZZ,NTL::vec_long> &f,
		  long n,
                  long pbound,
		  long bits_inf_norm );

/* Dense method using "best known technology", which is just
 * a Newton iteration. This is a modular algorithm, so it should
 * be quite fast.
 * pbound is used to choose primes (as above), and 
 * tms is the number of nonzero terms in the input polynomial 
 * (used to bound the power r using Schinzel's theorem).
 */
long pp_dense( const NTL::ZZX& f, long pbound, long tms );

/* Computes floor(log(infinity norm of f)) */
long bits_in_inf_norm(NTL::ZZX f);

/* Convert from the standard (dense) polynomial representation
 * to the lacunary representation of a list of pairs of
 * nonzero coefficient and exponent difference.
 * That is, a pair of vectors (c_1,c_2,...,c_t) and (e_1,e_2,...,e_t)
 * such that the given polynomial (in) could be written as
 * c_t*x^(e_t) + c_(t-1)*x^(e_(t-1)+e_t) + ... + c_1*x^(e_1+e_2+...+e_t)
 */
void denseToSparse(std::pair<NTL::vec_ZZ,NTL::vec_long> &out, 
                   const NTL::ZZX& in);

/* Compute the least number of bits in a random prime in order for
 * the dense and sparse algorithms to succeed with high probability.
 * This will be the number of bits in the random prime.
 * The bound is computed in terms of the size of the root h
 * (degree, number of nonzero terms, and bits in infinity norm),
 * and the power r to which h is raised. The input to the algorithm will
 * of course be h^r. We use the fact that the 2-norm of h^r is at most
 * the 1-norm of h to the power r, by an equivalent proof to that of
 * Theorem 2.9.
 * min_p_bits is the least number of bits that the prime must have, to
 * guarantee that the field is large enough and therefore certain 
 * theorems apply.
 */
long pbound( long deg_h, long tms_h, long bits_inf_norm_h, long r,
             long min_p_bits );

/* These two functions compute the minimal number of bits
 * in a valid prime for the lacunary (resp. dense) perfect power
 * algorithms when the input polynomial has the given degree.
 */
inline long lacunary_min_p_bits( long deg_f ) {
	// For the lacunary algorithm, we need p >= 4(n-1)^2
	return 2L + 2L * NTL::NumBits(deg_f-1);
}
inline long dense_min_p_bits( long deg_f ) {
	// For the dense algorithm, we need p >= 4n.
	return 2L + NTL::NumBits(deg_f);
}

// Constant for the number of primes to precompute
const long NUM_PRIMES = 4000;
// Constant to hold the list of the first NUM_PRIMES primes
extern long FIRST_PRIMES[NUM_PRIMES];
// Initializes FIRST_PRIMES
void init_primes();

// Sets a = gcd(a,b)
inline void gcd(long& a, long b) {
    long c = b % a;
    while(c) {
	b = a;
	a = c;
    	c = b % a;
    }
}

#endif // PP_H
